<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Game extends Model
{
    use HasFactory;

    /**
     * A tabela do banco de dados usada pelo modelo.
     *
     * @var string
     */
    protected $table = 'games';

    /**
     * A chave primária associada à tabela.
     *
     * @var string
     */
    protected $primaryKey = 'id';

    /**
     * Atributos que são atribuíveis em massa.
     *
     * @var array
     */
    protected $fillable = [
        'provider_id',
        'game_name',
        'game_id',
        'game_type',
        'cover',
        'status',
        'technology',
        'distribution',
        'views',
        'is_featured',
        'show_home',
        'game_code'
    ];

    /**
     * Atributos que são automaticamente adicionados ao modelo.
     *
     * @var array
     */
    protected $appends = [
        'hasFavorite',
        'totalFavorites',
        'hasLike',
        'totalLikes',
        'dateHumanReadable',
        'createdAt'
    ];

    /**
     * Retorna o número total de favoritos.
     *
     * @return int
     */
    public function getTotalFavoritesAttribute()
    {
        return $this->favorites()->count();
    }

    /**
     * Retorna o número total de likes.
     *
     * @return int
     */
    public function getTotalLikesAttribute()
    {
        return $this->likes()->count();
    }

    /**
     * Verifica se o usuário autenticado deu like no jogo.
     *
     * @return bool
     */
    public function getHasLikeAttribute()
    {
        if (auth('api')->check() && !empty($this->attributes['id'])) {
            return $this->likes()->where('user_id', auth('api')->id())->exists();
        }
        return false;
    }

    /**
     * Verifica se o usuário autenticado adicionou o jogo aos favoritos.
     *
     * @return bool
     */
    public function getHasFavoriteAttribute()
    {
        if (auth('api')->check() && !empty($this->attributes['id'])) {
            return $this->favorites()->where('user_id', auth('api')->id())->exists();
        }
        return false;
    }

    /**
     * Relacionamento com a tabela providers.
     *
     * @return BelongsTo
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(Provider::class, 'provider_id', 'id');
    }

    /**
     * Relacionamento com a tabela categories.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function categories(): \Illuminate\Database\Eloquent\Relations\BelongsToMany
    {
        return $this->belongsToMany(Category::class);
    }

    /**
     * Relacionamento com a tabela favorites.
     *
     * @return HasMany
     */
    public function favorites(): HasMany
    {
        return $this->hasMany(GameFavorite::class, 'game_id', 'id');
    }

    /**
     * Relacionamento com a tabela likes.
     *
     * @return HasMany
     */
    public function likes(): HasMany
    {
        return $this->hasMany(GameLike::class, 'game_id', 'id');
    }

    /**
     * Relacionamento com a tabela reviews.
     *
     * @return HasMany
     */
    public function reviews(): HasMany
    {
        return $this->hasMany(GameReview::class, 'game_id', 'id');
    }

    /**
     * Retorna a data de criação formatada.
     *
     * @return string
     */
    public function getCreatedAtAttribute()
    {
        return Carbon::parse($this->attributes['created_at'])->format('Y-m-d');
    }

    /**
     * Retorna a data de criação de forma legível.
     *
     * @return string
     */
    public function getDateHumanReadableAttribute()
    {
        return Carbon::parse($this->created_at)->diffForHumans();
    }
}
